/**
 *   QuercusEngine
 *   Copyright 2010 Stani Ryabenkiy
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along
 *   with this program; if not, write to the Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.   
 */

/**
 * Parts of this software are derived from the Quercus JSR-223 Script Engine
 * implementation (com.caucho.quercus.script.QuercusScriptEngine), the license
 * for which is reproduced below.
 */

/* 
 * Copyright (c) 1998-2008 Caucho Technology -- all rights reserved
 *
 * This file is part of Resin(R) Open Source
 *
 * Each copy or derived work must preserve the copyright notice and this
 * notice unmodified.
 *
 * Resin Open Source is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Resin Open Source is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE, or any warranty
 * of NON-INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Resin Open Source; if not, write to the
 *
 *   Free Software Foundation, Inc.
 *   59 Temple Place, Suite 330
 *   Boston, MA 02111-1307  USA
 *
 * @author Scott Ferguson
 */

package org.apache.bsf.engines.quercus;



import java.io.IOException;
import java.io.StringReader;
import java.io.Writer;

import javax.script.ScriptContext;
import javax.script.SimpleScriptContext;

import org.apache.bsf.BSFDeclaredBean;
import org.apache.bsf.BSFException;
import org.apache.bsf.util.BSFEngineImpl;

import com.caucho.quercus.Quercus;
import com.caucho.quercus.QuercusExitException;
import com.caucho.quercus.env.Env;
import com.caucho.quercus.env.Value;
import com.caucho.quercus.page.InterpretedPage;
import com.caucho.quercus.page.QuercusPage;
import com.caucho.quercus.parser.QuercusParser;
import com.caucho.quercus.program.QuercusProgram;
import com.caucho.vfs.NullWriteStream;
import com.caucho.vfs.ReadStream;
import com.caucho.vfs.ReaderStream;
import com.caucho.vfs.WriteStream;
import com.caucho.vfs.WriterStreamImpl;

/**
 * The QuercusEngine for Bean Scripting Framework (BSF) 2.4 allows using 
 * Quercus through the BSF API. 
 * 
 * 
 * @author stani
 *
 */
public class QuercusEngine extends BSFEngineImpl {

	private Quercus quercus;
	private ScriptContext context;


	public QuercusEngine() {
		this.context = new SimpleScriptContext();
		
		// initialize Quercus engine
		this.quercus = new Quercus();
		this.quercus.init();		
	}


	/**
	 * this is a minimal implementation, calling into the engine is not supported
	 */
	public Object call(Object object, String name, Object[] args)
			throws BSFException {
		throw new BSFException(BSFException.REASON_UNSUPPORTED_FEATURE,
				   "QuercusEngine currently does not support call()",
				   null);
	}

	/**
	 * partially derived from com.caucho.quercus.script.QuercusScriptEngine.eval() 
	 * 
	 */
	public Object eval(String source, int lineNo, int colNo, Object expr)
			throws BSFException {
				
		try {
			ReadStream reader = ReaderStream.open(new StringReader((String)expr));  
			QuercusProgram program = QuercusParser.parse(quercus, null, reader);
		
			Writer writer = context.getWriter();
		      
		      WriteStream out;

		      if (writer != null) {
		        WriterStreamImpl s = new WriterStreamImpl();
		        s.setWriter(writer);
		        WriteStream os = new WriteStream(s);
		        
		        os.setNewlineString("\n");

		        try {
		          os.setEncoding("iso-8859-1");
		        } catch (Exception e) {
		        }

		        out = os;
		      }
		      else
		        out = new NullWriteStream();

		      QuercusPage page = new InterpretedPage(program);

		      Env env = new Env(quercus, page, out, null, null);

		      env.setScriptContext(context);

		      // php/214c
		      env.start();
		      
		      Object result = null;
		      
		      try {
		        Value value = program.execute(env);
		        
		        if (value != null)
		          result = value.toJavaObject();
		      }
		      catch (QuercusExitException e) {
		        //php/2148
		      }
		      
		      out.flushBuffer();
		      out.free();

		      // flush buffer just in case
		      //
		      // jrunscript in interactive mode does not automatically flush its
		      // buffers after every input, so output to stdout will not be seen
		      // until the output buffer is full
		      //
		      // http://bugs.caucho.com/view.php?id=1914
		      writer.flush();
		      
		      return result;
		}
		catch (IOException e) {
			throw new BSFException(e.getMessage());
		}
	}

	public void declareBean(BSFDeclaredBean bean) throws BSFException {
		context.getBindings(ScriptContext.ENGINE_SCOPE).put(bean.name, bean.bean);
	}

	public void undeclareBean(BSFDeclaredBean bean) throws BSFException {
		context.getBindings(ScriptContext.ENGINE_SCOPE).remove(bean.name);
	}

	
}
